#!/usr/bin/env python
# coding: utf-8

# In[1]:


### This program calculates mean lipid fluoresence of germinating zoospores using circular ROIs established from settled cells

### Usage: Run in Jupyter Notebook

### Author: Davis Laundon, The Marine Biological Association, September 2020 

#------------------------------------------------------------------------------------------------------------#
### Import the necessary packages before running the script
import os 
import glob

import skimage 
import matplotlib.pyplot as plt
import numpy as np
import scipy

from skimage import io
from skimage import morphology
from skimage.morphology import reconstruction
from skimage import filters
from scipy import ndimage as ndi
from skimage import measure, feature, morphology
from scipy.ndimage import distance_transform_edt

#------------------------------------------------------------------------------------------------------------#
### Load up the images as the initial bright-field image and fluoresence list
bf_image_list = np.sort(glob.glob('#######'))
fluo_image_list = np.sort(glob.glob('######'))

#------------------------------------------------------------------------------------------------------------#
### Set the manual parameters
ROI_radius = 5 ### Pixel diameter of the cell ROI 
Time_Frame = 25 ### Middle frame for mask generation
#------------------------------------------------------------------------------------------------------------#
### Create the initial mask for fluorescent quantification 
bf_frame = io.imread(bf_image_list[Time_Frame]) ### Take from the middle of the time series to allow for drift

### Invert the image for segmentation
bf_invert=np.invert(bf_frame)

### Convert images to 64 floats 
bf_float = skimage.img_as_float64(bf_invert)

### Background subtract using a dilation subtraction
seed = np.copy(bf_float) # Set seeds
seed[1:-1, 1:-1] =  bf_float.min()
mask =  bf_float
dilated = reconstruction(seed, mask, method='dilation') ### Create dilated image
bf_foreground=(bf_float - dilated) ### Generate foreground by subtraction 
    
bf_blurred = filters.gaussian(bf_foreground, sigma = 3) ### Blur image
    
bf_edges = filters.sobel(bf_blurred) ### Find edges
    
threshold = filters.threshold_otsu(bf_edges) ### Threshold out the edges into a binary mask

binary_cells = bf_edges > threshold ### Binarize
    
closed_cells = morphology.binary_closing(binary_cells) ### Close the edges
    
dilated_cells = morphology.binary_dilation(closed_cells) ### Dilate the cells
    
filled_cells = ndi.binary_fill_holes(dilated_cells) ### Fill the cells 

circles=skimage.transform.hough_circle(filled_cells, ROI_radius) ### Create circular masks around the cells of defined radius

stripped_circles = np.squeeze(circles) ### Remove a single dimension from the output circles

filled_circles = ndi.binary_fill_holes(stripped_circles) ### Fill the circles

circle_mask = filled_circles.astype('int32') ### Convert to integer image

#------------------------------------------------------------------------------------------------------------#
### Quantify the fluorescence 
fluo_results = [] ### Create an empty array to store the fluorescence data in

for i in range(len(fluo_image_list)): ### Set the index for the loop
    
    fluo_image = io.imread(fluo_image_list[i]) ### Load the fluorescent images
    
    
    fluo_float = skimage.img_as_float64(fluo_image) ### Convert fluo image to float
    
    ### Get fluo foreground image via background subtraction
    seed = np.copy(fluo_float)
    seed[1:-1, 1:-1] = fluo_float.min()
    mask = fluo_float
    dilated = reconstruction(seed, mask, method='dilation')
    fluo_foreground = (fluo_float - dilated)
    
    ### Measure the cells
    cell_IDs = skimage.measure.regionprops(circle_mask, fluo_foreground, cache=False) ### Overlay mask and fluoresence
    cell_intensity = [r.mean_intensity for r in cell_IDs] ### Measure mean intensity
    roi_area = [r.area for r in cell_IDs] ### Measure cell areas
    total_intensity = np.convolve(roi_area, cell_intensity) ### Get total intensity by area * mean intensity
        
    ### Add the fluo data to the empty array above 
    fluo_results.append(total_intensity)
    
    ### Generate a visual output of the cells and their ROIs
    fig, ax = plt.subplots(figsize=(15, 15)) ### Make the plot
    plt.imshow(circle_mask, cmap='tab20') ### Add the mask
    plt.imshow(fluo_foreground, alpha=0.9, cmap = 'Reds') ### Add the fluo image
    ax.axis('off')
    plt.savefig("#####" %i, bbox_inches='tight', pad_inches=0)
    
    ### Show loop progress
    print(i)
#------------------------------------------------------------------------------------------------------------#
### Turn data into a percentage of the initial starting fluorescence
percentage_array = [] ### Make an empty array
for n in range(len(fluo_results)): ### Loop over the fluoresence values
    percentage= ((fluo_results[n]/fluo_results[0])*100) ### Convert fluo value to a percentage of T0
    percentage_array.append(percentage) ### Append data to empty array

### Save the fluo_data to csv 
np.savetxt('######', percentage_array, delimiter=',', header='percentage_intensity')

### Confirm the script is complete 
print('SCRIPT DONE')

